"""Sensor platform for EKZ Tariff integration."""
from __future__ import annotations

import logging
from typing import Any

from homeassistant.components.sensor import (
    SensorDeviceClass,
    SensorEntity,
    SensorStateClass,
)
from homeassistant.config_entries import ConfigEntry
from homeassistant.core import HomeAssistant
from homeassistant.helpers.entity_platform import AddEntitiesCallback
from homeassistant.helpers.update_coordinator import CoordinatorEntity

from .const import (
    DOMAIN,
    ATTR_FORECAST,
    ATTR_TOMORROW_AVAILABLE,
    ATTR_CURRENT_SLOT_START,
    ATTR_CURRENT_SLOT_END,
)
from .coordinator import EKZTariffCoordinator

_LOGGER = logging.getLogger(__name__)

# Einheit: Rappen pro kWh
UNIT_RP_KWH = "Rp./kWh"
# Einheit: CHF pro kWh  
UNIT_CHF_KWH = "CHF/kWh"


async def async_setup_entry(
    hass: HomeAssistant,
    entry: ConfigEntry,
    async_add_entities: AddEntitiesCallback,
) -> None:
    """Set up EKZ Tariff sensors based on a config entry."""
    coordinator: EKZTariffCoordinator = hass.data[DOMAIN][entry.entry_id]
    
    entities = [
        # Current prices
        EKZCurrentPriceSensor(
            coordinator,
            "current_electricity_price",
            "EKZ Energiepreis",
            "mdi:lightning-bolt",
        ),
        EKZCurrentPriceSensor(
            coordinator,
            "current_grid_price",
            "EKZ Netzpreis",
            "mdi:transmission-tower",
        ),
        EKZCurrentPriceSensor(
            coordinator,
            "current_integrated_price",
            "EKZ Gesamtpreis",
            "mdi:currency-usd",
        ),
        
        # Today's statistics - Integrated
        EKZStatisticSensor(
            coordinator,
            "today_integrated_min",
            "EKZ Gesamtpreis Heute Min",
            "mdi:arrow-down",
        ),
        EKZStatisticSensor(
            coordinator,
            "today_integrated_max",
            "EKZ Gesamtpreis Heute Max",
            "mdi:arrow-up",
        ),
        EKZStatisticSensor(
            coordinator,
            "today_integrated_avg",
            "EKZ Gesamtpreis Heute Durchschnitt",
            "mdi:approximately-equal",
        ),
        
        # Forecast sensor with full data
        EKZForecastSensor(coordinator),
        
        # Price level indicator (cheap/normal/expensive)
        EKZPriceLevelSensor(coordinator),
    ]
    
    async_add_entities(entities)


class EKZBaseSensor(CoordinatorEntity[EKZTariffCoordinator], SensorEntity):
    """Base class for EKZ Tariff sensors."""
    
    _attr_has_entity_name = True
    
    def __init__(
        self,
        coordinator: EKZTariffCoordinator,
        key: str,
        name: str,
        icon: str,
    ) -> None:
        """Initialize the sensor."""
        super().__init__(coordinator)
        self._key = key
        self._attr_name = name
        self._attr_icon = icon
        self._attr_unique_id = f"ekz_tariff_{key}"
        self._attr_device_info = {
            "identifiers": {(DOMAIN, "ekz_tariff")},
            "name": "EKZ Stromtarife",
            "manufacturer": "EKZ",
            "model": "Dynamischer Stromtarif",
        }


class EKZCurrentPriceSensor(EKZBaseSensor):
    """Sensor for current electricity price."""
    
    _attr_native_unit_of_measurement = UNIT_RP_KWH
    _attr_device_class = SensorDeviceClass.MONETARY
    _attr_state_class = SensorStateClass.MEASUREMENT
    _attr_suggested_display_precision = 2
    
    @property
    def native_value(self) -> float | None:
        """Return the current price in Rappen/kWh."""
        if self.coordinator.data is None:
            return None
        
        value = self.coordinator.data.get(self._key)
        if value is not None:
            # API liefert CHF/kWh, wir zeigen Rappen/kWh
            return round(value * 100, 2)
        return None
    
    @property
    def extra_state_attributes(self) -> dict[str, Any]:
        """Return additional state attributes."""
        if self.coordinator.data is None:
            return {}
        
        attrs = {}
        
        # Price in CHF/kWh as additional attribute
        value = self.coordinator.data.get(self._key)
        if value is not None:
            attrs["price_chf_kwh"] = round(value, 5)
        
        slot_start = self.coordinator.data.get("current_slot_start")
        slot_end = self.coordinator.data.get("current_slot_end")
        
        if slot_start:
            attrs[ATTR_CURRENT_SLOT_START] = slot_start.isoformat()
        if slot_end:
            attrs[ATTR_CURRENT_SLOT_END] = slot_end.isoformat()
        
        attrs[ATTR_TOMORROW_AVAILABLE] = self.coordinator.data.get(
            "tomorrow_available", False
        )
        
        return attrs


class EKZStatisticSensor(EKZBaseSensor):
    """Sensor for price statistics (min/max/avg)."""
    
    _attr_native_unit_of_measurement = UNIT_RP_KWH
    _attr_device_class = SensorDeviceClass.MONETARY
    _attr_state_class = SensorStateClass.MEASUREMENT
    _attr_suggested_display_precision = 2
    
    @property
    def native_value(self) -> float | None:
        """Return the statistic value in Rappen/kWh."""
        if self.coordinator.data is None:
            return None
        
        value = self.coordinator.data.get(self._key)
        if value is not None:
            return round(value * 100, 2)
        return None


class EKZForecastSensor(CoordinatorEntity[EKZTariffCoordinator], SensorEntity):
    """Sensor providing full forecast data as attributes."""
    
    _attr_has_entity_name = True
    _attr_name = "EKZ Preisprognose"
    _attr_icon = "mdi:chart-timeline-variant"
    
    def __init__(self, coordinator: EKZTariffCoordinator) -> None:
        """Initialize the forecast sensor."""
        super().__init__(coordinator)
        self._attr_unique_id = "ekz_tariff_forecast"
        self._attr_device_info = {
            "identifiers": {(DOMAIN, "ekz_tariff")},
            "name": "EKZ Stromtarife",
            "manufacturer": "EKZ",
            "model": "Dynamischer Stromtarif",
        }
    
    @property
    def native_value(self) -> int | None:
        """Return number of forecast entries."""
        if self.coordinator.data is None:
            return None
        
        forecast = self.coordinator.data.get("forecast", [])
        return len(forecast)
    
    @property
    def extra_state_attributes(self) -> dict[str, Any]:
        """Return the forecast data as attributes."""
        if self.coordinator.data is None:
            return {}
        
        forecast = self.coordinator.data.get("forecast", [])
        
        # Convert prices to Rappen for easier reading
        forecast_rp = []
        for item in forecast:
            forecast_rp.append({
                "start": item["start"],
                "end": item["end"],
                "electricity_rp": round(item["electricity_price"] * 100, 2) if item.get("electricity_price") else None,
                "grid_rp": round(item["grid_price"] * 100, 2) if item.get("grid_price") else None,
                "total_rp": round(item["integrated_price"] * 100, 2) if item.get("integrated_price") else None,
            })
        
        return {
            ATTR_FORECAST: forecast_rp,
            ATTR_TOMORROW_AVAILABLE: self.coordinator.data.get(
                "tomorrow_available", False
            ),
            "publication_timestamp": self.coordinator.data.get("publication_timestamp"),
            "last_update": self.coordinator.data.get("last_update", "").isoformat()
            if self.coordinator.data.get("last_update")
            else None,
        }


class EKZPriceLevelSensor(CoordinatorEntity[EKZTariffCoordinator], SensorEntity):
    """Sensor indicating if current price is cheap, normal, or expensive."""
    
    _attr_has_entity_name = True
    _attr_name = "EKZ Preisniveau"
    _attr_icon = "mdi:speedometer"
    
    def __init__(self, coordinator: EKZTariffCoordinator) -> None:
        """Initialize the price level sensor."""
        super().__init__(coordinator)
        self._attr_unique_id = "ekz_tariff_price_level"
        self._attr_device_info = {
            "identifiers": {(DOMAIN, "ekz_tariff")},
            "name": "EKZ Stromtarife",
            "manufacturer": "EKZ",
            "model": "Dynamischer Stromtarif",
        }
    
    @property
    def native_value(self) -> str | None:
        """Return the price level."""
        if self.coordinator.data is None:
            return None
        
        current = self.coordinator.data.get("current_integrated_price")
        avg = self.coordinator.data.get("today_integrated_avg")
        
        if current is None or avg is None:
            return "unknown"
        
        # Calculate thresholds
        # Cheap: below 80% of daily average
        # Expensive: above 120% of daily average
        cheap_threshold = avg * 0.8
        expensive_threshold = avg * 1.2
        
        if current <= cheap_threshold:
            return "günstig"
        elif current >= expensive_threshold:
            return "teuer"
        else:
            return "normal"
    
    @property
    def extra_state_attributes(self) -> dict[str, Any]:
        """Return additional attributes."""
        if self.coordinator.data is None:
            return {}
        
        current = self.coordinator.data.get("current_integrated_price")
        avg = self.coordinator.data.get("today_integrated_avg")
        
        if current is None or avg is None:
            return {}
        
        return {
            "current_price_rp": round(current * 100, 2),
            "avg_price_rp": round(avg * 100, 2),
            "cheap_threshold_rp": round(avg * 0.8 * 100, 2),
            "expensive_threshold_rp": round(avg * 1.2 * 100, 2),
            "current_price_chf": round(current, 5),
            "avg_price_chf": round(avg, 5),
        }
