"""Data coordinator for EKZ Tariff integration."""
from __future__ import annotations

import logging
from datetime import datetime, timedelta
from typing import Any

import aiohttp
from homeassistant.core import HomeAssistant
from homeassistant.helpers.update_coordinator import (
    DataUpdateCoordinator,
    UpdateFailed,
)

from .api import EKZTariffAPI, EKZTariffAPIError
from .const import DOMAIN, UPDATE_INTERVAL_MINUTES

_LOGGER = logging.getLogger(__name__)


class EKZTariffCoordinator(DataUpdateCoordinator[dict[str, Any]]):
    """Coordinator to manage fetching EKZ tariff data."""
    
    def __init__(
        self,
        hass: HomeAssistant,
        session: aiohttp.ClientSession,
    ) -> None:
        """Initialize the coordinator."""
        super().__init__(
            hass,
            _LOGGER,
            name=DOMAIN,
            update_interval=timedelta(minutes=UPDATE_INTERVAL_MINUTES),
        )
        
        self._api = EKZTariffAPI(session)
    
    async def _async_update_data(self) -> dict[str, Any]:
        """Fetch data from EKZ API."""
        try:
            # Fetch tariff data
            data = await self._api.get_tariffs(use_cache=True, cache_hours=1)
            
            prices = data.get("prices", [])
            now = datetime.now().astimezone()
            today = now.date()
            tomorrow = today + timedelta(days=1)
            
            # Get current price
            current_price = self._api.get_current_price(prices)
            
            # Get today's prices for statistics
            today_prices = self._api.get_prices_for_date(prices, now)
            
            # Check if tomorrow's data is available
            tomorrow_prices = self._api.get_prices_for_date(
                prices, 
                now + timedelta(days=1)
            )
            tomorrow_available = len(tomorrow_prices) > 0
            
            # Calculate statistics for integrated price (Strom + Netz)
            today_stats = self._api.calculate_statistics(today_prices, "integrated_price")
            today_electricity_stats = self._api.calculate_statistics(today_prices, "electricity_price")
            today_grid_stats = self._api.calculate_statistics(today_prices, "grid_price")
            
            # Build forecast data (future prices)
            forecast = self._build_forecast(prices, now)
            
            return {
                # Current prices (in CHF/kWh)
                "current_electricity_price": current_price["electricity_price"] if current_price else None,
                "current_grid_price": current_price["grid_price"] if current_price else None,
                "current_integrated_price": current_price["integrated_price"] if current_price else None,
                "current_total_price": current_price["total_price"] if current_price else None,
                
                # Current slot info
                "current_slot_start": current_price["start"] if current_price else None,
                "current_slot_end": current_price["end"] if current_price else None,
                
                # Today's statistics - Integrated (Strom + Netz)
                "today_integrated_min": today_stats["min"],
                "today_integrated_max": today_stats["max"],
                "today_integrated_avg": today_stats["avg"],
                
                # Today's statistics - Electricity only
                "today_electricity_min": today_electricity_stats["min"],
                "today_electricity_max": today_electricity_stats["max"],
                "today_electricity_avg": today_electricity_stats["avg"],
                
                # Today's statistics - Grid only
                "today_grid_min": today_grid_stats["min"],
                "today_grid_max": today_grid_stats["max"],
                "today_grid_avg": today_grid_stats["avg"],
                
                # Tomorrow availability
                "tomorrow_available": tomorrow_available,
                
                # Full forecast data
                "forecast": forecast,
                
                # Raw prices for advanced usage
                "prices": prices,
                
                # Publication info
                "publication_timestamp": data.get("publication_timestamp"),
                
                # Last update
                "last_update": now,
            }
            
        except EKZTariffAPIError as err:
            raise UpdateFailed(f"Error fetching EKZ tariff data: {err}") from err
    
    def _build_forecast(
        self,
        prices: list[dict[str, Any]],
        now: datetime,
    ) -> list[dict[str, Any]]:
        """Build forecast data for future prices."""
        forecast = []
        
        for price in prices:
            if price["end"] <= now:
                continue  # Skip past prices
            
            forecast.append({
                "start": price["start"].isoformat(),
                "end": price["end"].isoformat(),
                "electricity_price": round(price["electricity_price"], 5) if price["electricity_price"] else None,
                "grid_price": round(price["grid_price"], 5) if price["grid_price"] else None,
                "integrated_price": round(price["integrated_price"], 5) if price["integrated_price"] else None,
                "total_price": round(price["total_price"], 5) if price["total_price"] else None,
            })
        
        return forecast
