"""Config flow for EKZ Tariff integration."""
from __future__ import annotations

import logging
from typing import Any

from homeassistant import config_entries
from homeassistant.core import HomeAssistant
from homeassistant.helpers.aiohttp_client import async_get_clientsession

from .api import EKZTariffAPI, EKZTariffAPIError
from .const import DOMAIN

_LOGGER = logging.getLogger(__name__)


async def validate_api(hass: HomeAssistant) -> bool:
    """Validate that the API is reachable."""
    session = async_get_clientsession(hass)
    api = EKZTariffAPI(session)
    
    try:
        data = await api.get_tariffs(use_cache=False)
        return len(data.get("prices", [])) > 0
    except EKZTariffAPIError as err:
        _LOGGER.error("API validation failed: %s", err)
        return False


class EKZTariffConfigFlow(config_entries.ConfigFlow, domain=DOMAIN):
    """Handle a config flow for EKZ Tariff."""
    
    VERSION = 1
    
    async def async_step_user(
        self, 
        user_input: dict[str, Any] | None = None
    ) -> config_entries.FlowResult:
        """Handle the initial step."""
        errors: dict[str, str] = {}
        
        # Only allow one instance
        await self.async_set_unique_id(DOMAIN)
        self._abort_if_unique_id_configured()
        
        if user_input is not None:
            # Validate API connection
            if await validate_api(self.hass):
                return self.async_create_entry(
                    title="EKZ Stromtarife",
                    data={},
                )
            else:
                errors["base"] = "cannot_connect"
        
        # Show confirmation form
        return self.async_show_form(
            step_id="user",
            errors=errors,
        )
